% --------------------------------------------------------------
% Set up matrix A and right-hand side s
% --------------------------------------------------------------
function [A, s] = setAs(a, b, c, d, n, m)

% Arguments:
%    a    =  width of inner conductor
%    b    = height of inner conductor
%    c    =  width of outer conductor
%    d    = height of outer conductor
%    n    = number of points in the x-direction (horizontal)
%    m    = number of points in the y-direction (vertical)
% Returns:
%    A    = matrix on sparse storage format
%    s    = right-hand side on sparse storage format

hx = 0.5*c/n;           % Grid size in x-direction
na = round(0.5*a/hx);   % Number of cells for half width of 
                        % inner conductor
hy = 0.5*d/m;           % Grid size in y-direction
m  = round(0.5*d/hy);   % Number of cells for half height of 
                        % outer conductor
mb = round(0.5*b/hy);   % Number of cells for half height of 
                        % inner conductor
p  = 1;                 % Potential on inner conductor

%          The upper right corner is discretized
%
%               --------------------+
%                        c/2        |
%                                   |
%               ------------+       | d/2      
%                   a/2     |       |
%                           | b/2   |
%                           |       |
%
%                     (Dimensions)
%
%
%             The nodes are numbered like this
%        y
%        ^
%        |
% (m-1)hy| (m-1)n+1 (m-1)n+2 (m-1)n+3  ...    mn
%        |     :        :        :            :
%   2hy  |   2n+1     2n+2     2n+3    ...    3n   
%    hy  |    n+1      n+2      n+3    ...    2n
%     0  |     1        2        3     ...     n
%           --------------------------------------> x
%              0       hx      2 hx        (n-1)hx
%
%                   (Discretization)

N  = n * m;     % Total number of unknowns.
cx = hx^-2;
cy = hy^-2;

% Generate a matrix with N = m*n rows (-> nodes on the grid),
% and five columns, one for each nonzero diagonal of A.
% The first column gives contribution from nodes beneath.
% The second column gives contribution from nodes to the left.
% The third column gives self-contribution.
% The fourth column gives contribution from nodes to the right.
% The fifth column gives contribution from nodes above.

% The following lines assume some knowledge of MATLAB.  If you 
% feel uncertain, insert the 'keyboard' command.  This causes 
% MATLAB to stop.  Then execute lines by 'dbstep' and examine 
% the result.

C = repmat([cy cx -2*(cx+cy) cx cy], N, 1);

% Find indices of nodes that are not surrounded by four interior 
% nodes.
idx0R = n:n:N-n;          % Nodes with     V = 0 to the right
idxNB = na+2:n;           % Nodes with dV/dy = 0 beneath
idxNL = 1+n*(mb+1):n:N;   % Nodes with dV/dx = 0 to the left

idx1C = repmat((1:na+1)', 1, mb+1) + repmat((0:n:n*mb),na+1,1);
                          % 'x-index + n*(y-index-1)' for all
idx1C = idx1C(:)';        % nodes on (or inside) the inner 
                          % conductor where V = 1  
                          % and convert to row vector

C(idx1C,[1 2 4 5]) = 0;
C(idx1C, 3) = 1;          
C(idx0R, 4) = 0;
C(idxNB, 5) = 2*cy;
C(idxNL, 4) = 2*cx;
C(idxNL, 2) = 0;

% Find the nonzero elements (si) of each column and the
% corresponding row indices (ii).  Do not include elements
% corresponding to nodes outside the grid.
[i1,j,s1] = find(C(n+1:end,   1)); % The first 'nc' nodes have no
                                   % neighbors beneath
[i2,j,s2] = find(C(1+1:end,   2)); % The first node has no
                                   % neighbor to the left
[i3,j,s3] = find(C(  1:end,   3));
[i4,j,s4] = find(C(  1:end-1, 4)); % The last node has no
                                   % neighbor to the right
[i5,j,s5] = find(C(  1:end-n, 5)); % The last 'nc' nodes have no
                                   % neighbors above

% Put the elements (si) into a sparse matrix. The first input 
% are row indices, the second is column indices and the third 
% is the elements.
A = sparse([i1+n; i2+1; i3; i4; i5], ...
           [i1; i2; i3; i4+1; i5+n], ...
           [s1; s2; s3; s4; s5], N, N);
s = sparse(idx1C', 1, p, N, 1);
